/**
 *
 * \file        ethernet_console.c
 *
 * \brief       Implementation of a console works over Ethernet/TCP/IP.
 *
 * \author      Pete McCormick
 *
 * \date        4/24/2008
 *
 * \note        This is a generic file.  There should be NO product-specific
 *              nor processor-specific stuff in here.
 *
 */

////////////////////////////////////////////////////////////////////////////////

#include "console.h"
#include "os.h"
#include "string_utils.h"
#include <string.h>
#include "leds.h"
#include "hardware.h"
#include "ethernet.h"
#include "device.h"
#include <stdio.h>
#include "errors.h"

////////////////////////////////////////////////////////////////////////////////

#define ETH_CONSOLE_MAX_CONNECTIONS     1
#define ETH_CONSOLE_TCP_PORT            41795
#define ETHCONSOLE_LOCKTIMEOUT          20000

extern ETHER_PARAM *pEtherParam;   // Ethernet Parameters
PTNQXMODPKT pfPTNQXmodemPacket;
PTSCKTCLSD pfPTSocketClosed;//process socket closure
PF_DM_ETHSW_HW_RESET g_pEthernetSwitchHardwareResetFunc = 0;

////////////////////////////////////////////////////////////////////////////////

typedef struct
{
    //UINT32 bps;
    CMDINPUT cmdInput[ETH_CONSOLE_MAX_CONNECTIONS+1];
    UINT32   socket[ETH_CONSOLE_MAX_CONNECTIONS+1];   // 1-based
    BOOL     used[ETH_CONSOLE_MAX_CONNECTIONS+1];   // 1-based
    UINT32   writeLock[ETH_CONSOLE_MAX_CONNECTIONS+1];   // 1-based
}ETHCONSOLE;

////////////////////////////////////////////////////////////////////////////////

#ifdef __cplusplus
extern "C" void EthConsoleInit(void);
extern "C" INT32 EthernetDeviceProcessInput(UINT32 inst, UINT32 cmd, UINT8 * pData, UINT32 * pDataBytes);
#endif

static ETHCONSOLE EthConsole;

BOOL EthDummy;

////////////////////////////////////////////////////////////////////////////////

void EthConsoleDoNothing(void)
{
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Unused for now - needed if we ever implement
 *              XGET, XPUT, or SYSTEM commands.
 *
 * \param       void
 *
 */
void EthConsoleCaptureInput(void)
{
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Unused for now - needed if we ever implement
 *              XGET, XPUT, or SYSTEM commands.
 *
 * \param       void
 *
 */
void EthConsoleReleaseInput(void)
{
}

void EthConsoleTx(UINT8 * pSrc, UINT32 byteCnt, UINT32 inst)
{
    if(EthConsole.used[inst])
    {
        // MNT - 8.4.08 - Circular reference to DmConsolePrintf - Blows the stack
        //DmConsolePrintf("EthConsoleTx, inst=%lu\n", inst);
        if (OsLockWait(EthConsole.writeLock[inst], ETHCONSOLE_LOCKTIMEOUT) == -1)
        {
            DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_CONSOLE_LOCK_FAILED,0);
            return;
        } //if (OsLockWait(EthConsole.writeLock[i], ETHCONSOLE_LOCKTIMEOUT) == -1)

        // make sure the socket is valid
        if(EthConsole.used[inst])
        {
            if (NetSendTcpData(EthConsole.socket[inst], pSrc, byteCnt) < 0)
                DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_CONSOLE_LOCK_FAILED,2);
        }
        else
        {
            DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_CONSOLE_LOCK_FAILED,1);
        }

        OsUnlock(EthConsole.writeLock[inst]);
    }
}
/**
 * \author      Andrew Salmon
 *
 * \date        08/03/2011
 *
 * \return      1 if write lock was done
 *
 * \retval      UINT8
 *
 * \brief       Lock Console write indefinitely
 *
 *
 * \param       param console instance
 *
 */
UINT8 EthGetConsoleLock(UINT32 param, UINT8 lock)
{
    if( (param <= ETH_CONSOLE_MAX_CONNECTIONS) && EthConsole.writeLock[param])
    {
        if(lock )
        {
            if(EthConsole.used[param])
            {
                OsLock(EthConsole.writeLock[param]);
                return 1;
            }
        }
        else
        {
            OsUnlock(EthConsole.writeLock[param]);
        }
    }
    return 0 ;
}
// callback from net stack when a new connection is made
static INT32 EthConsoleNewConnection(UINT32 param, UINT32 remoteIp,
                                     UINT16 remotePort, UINT32 socket)
{
    UINT32 inst;
    CMDINPUT * pCmdInput;
    BOOL foundOne = 0;
    UINT32 i;

/*
    if(inst > ETH_CONSOLE_MAX_CONNECTIONS)
    {
        return -1;
    }
*/

    // MNT - 9/3/2008 - We don't care about the Tcp Helper Allocation here.
    // Find a free console instance and use it
    for(i=0; i<ETH_CONSOLE_MAX_CONNECTIONS; i++)
    {
        if(!EthConsole.used[i])
        {
            inst = i;
            EthConsole.used[i] = 1;
            foundOne = 1;
            break;
        }
    } // for(i=0; i<ETH_CONSOLE_MAX_CONNECTIONS; i++)

    if(!foundOne)
    {
      // MNT - 9/3/2008 - This should never happen - we should never reach here if the connection
      // is never being created by the TCP Read Task
      return -1;
    } // if(!foundOne)

    if (!EthConsole.writeLock[i])
      {
        OsCreateLock(&(EthConsole.writeLock[i]));
      } //(!EthConsole.writeLock[i])

    pCmdInput = &EthConsole.cmdInput[inst];
    memset(pCmdInput, 0, sizeof(*pCmdInput));
    // needed when actually parsing the command
    pCmdInput->command.pCmdInput = pCmdInput;
    pCmdInput->pSendFunc = EthConsoleTx;
    pCmdInput->sendParam = inst;
    pCmdInput->pCaptureFunc = EthConsoleCaptureInput;
    pCmdInput->pReleaseFunc = EthConsoleReleaseInput;
    pCmdInput->cmdSource = FROM_CTP_CLIENT;
    pCmdInput->printLock = 0;
    pCmdInput->TjiPrintLock = 0;
    pCmdInput->consoleLock = 0;
    pCmdInput->captured  = 0;
    pCmdInput->pPassto = NULL;

    CmdInputAddToList(pCmdInput);

	EthConsole.socket[inst] = socket;
    DmConsoleDisplayWelcome(pCmdInput);

	#ifdef DEBUG
    DmConsolePrintf("New connection on socket %lu, inst=%lu\n", socket, inst);
	#endif
    return inst;
}


/**
 *
 * \date        09-03-2008
 * \author      mtalreja
 * \brief       EthernetConsoleDisconnection
 * \detail      Set the instance as free.
 * \return
 * \retval
 * \warning
 * \note
 *
 * \param param
 * \return
 */
static INT32 EthernetConsoleDisconnection(UINT32 param)
  {
    CMDINPUT * pCmdInput;

    if (param<ETH_CONSOLE_MAX_CONNECTIONS)
      {
        pCmdInput = &EthConsole.cmdInput[param];
        // Xmodem capture mode.
        if ( pCmdInput->captured )
        {
           if(pfPTSocketClosed)
           {
              (*pfPTSocketClosed)();//hang up modem
           }
        }

        if (pCmdInput->pPassto)
        {
            pCmdInput->pPassto->pPassto = NULL;
        }
        EthConsole.used[param] = 0;
      }
    return 0; // success
  }


// callback from net stack when some bytes are received on our socket
static INT32 EthConsoleRxData(UINT32 param, UINT8 * pData, UINT32 dataBytes)
{
    UINT32 inst = param;
    CMDINPUT * pCmdInput;

    pCmdInput = &EthConsole.cmdInput[inst];

    // Xmodem capture mode.
    if ( pCmdInput->captured )
    {
        //check if this ethernet console supports xmodem captures
        if ( pfPTNQXmodemPacket )
        {
            (*pfPTNQXmodemPacket)((unsigned char*)pData, dataBytes);
            return 0;
        }
    }

   if (pCmdInput->pPassto)
   {
       pCmdInput->pPassto->pSendFunc(pData, dataBytes, pCmdInput->pPassto->sendParam);
       return 0;
   }

    // make sure last command has been processed since buffer is shared
    if ( DmConsoleWaitBufferAvailable(pCmdInput, MAX_CONSOLE_WAIT) )
    {
        while (dataBytes--)
        {
            // buffer the input;  If EOL received, sends command to Console task
            DmConsoleBufferInput(pCmdInput, *pData++);
        }
    }

    if (pCmdInput->killMe)
    {
        return -999;
    }

    return 0;
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Sets up structures used for the UART console.
 *
 * \param       void
 *
 * \note        Runs once at startup time.
 *
 */
void EthConsoleInit(void)
{
    INT32 result;

//     CMDINPUT * pCmdInput = &EthConsole.cmdInput;
//
//     // needed when actually parsing the command
//     pCmdInput->command.pCmdInput = pCmdInput;
//     pCmdInput->pSendFunc = EthConsoleTx;
//     pCmdInput->sendParam = 0;
//     pCmdInput->pCaptureFunc = EthConsoleCaptureInput;
//     pCmdInput->pReleaseFunc = EthConsoleReleaseInput;
//     pCmdInput->cmdSource = FROM_CTP_CLIENT;
//
//     CmdInputAddToList(pCmdInput);

    // MNT - 3/9/2009 - Use the port specified by the user. Bugzilla bug 27213
    // http://linux1/bugzilla/show_bug.cgi?id=27213
    result = NetListenOnTcpPort(pEtherParam->CTP_PortNum, ETH_CONSOLE_MAX_CONNECTIONS,
                       EthConsoleNewConnection, EthernetConsoleDisconnection, EthConsoleRxData, 0);

    if (result < 0)
      DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_CONSOLE_INIT,1);

    // wait upto 1 second for the initialization of port
    int timeout = 1000;
    while(!NetInitComplete() && timeout--)
    {
        HwDelayMsec(1);
    }
    if (timeout <= 0)
      DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_CONSOLE_INIT,0);

}

/**
 * \author      Pete McCormick
 *
 * \date        3/25/2008
 *
 * \return      INT32
 *
 * \retval      0 success
 * \retval      <0 failure
 *
 * \brief       Implement the specified device command
 *
 * \param       inst - unused
 * \param       cmd
 * \param       pData
 * \param       pDataBytes
 *
 * \note
 *
 */
INT32 EthernetDeviceProcessInput(UINT32 inst, UINT32 cmd, UINT8 * pData, UINT32 * pDataBytes)
{
#ifndef STM32F2XX
    UINT32 byteCnt;
#endif

    switch(cmd)
    {
        case DEVICE_INIT:
            // turn phy on or off
            //EthPhyCtrl(*pData);
            break;

        case DEVICE_TX: // for comm devices
            break;

        case DEVICE_RX: // for comm devices
            return DEVICE_OUTPUT_RAW;

        case DEVICE_PRINT:
            // dump everything (or something) in ascii hex
            // could use first byte to indicate which thing to print
            sprintf((char *)pData, "Hello, I am ethernet\r\n");
            return DEVICE_OUTPUT_ASCII;

        case DEVICE_SET_DEBUG:
            // set my own debug level
            //pI2C->debugLevel = *pData;
            break;

        case DEVICE_GET_DEBUG:
            // get the current debug level
            //*pData = pI2C->debugLevel;
            return DEVICE_OUTPUT_RAW;

        case DEVICE_SET:
            #ifndef STM32F2XX
        // syntax: dev eth set addr data data data ...
            byteCnt = *pDataBytes - 1;
            EthPhyWriteBlock(1, pData[0], &pData[1], byteCnt);
            #endif
            break;

        case DEVICE_GET:
        // syntax: dev eth get addr bytecnt
            #ifndef STM32F2XX
            byteCnt = pData[1];
            EthPhyReadBlock(1, pData[0], pData, pData[1]);
            *pDataBytes = byteCnt;
            #endif
            return DEVICE_OUTPUT_RAW;

            // user stuff:
        case DEVICE_SELFTEST:
            break;

        case DEVICE_RESET:

            if(g_pEthernetSwitchHardwareResetFunc)
               g_pEthernetSwitchHardwareResetFunc();
            else
               DmConsolePrintf("dev eth reset not supported\r");

            break;

        case DEVICE_SYNTAX:
            DmConsolePrintf("dev eth init 0\r");
            DmConsolePrintf("dev eth init 1\r");
            DmConsolePrintf("dev eth set {addr} {data} [data] [data] ...\r");
            DmConsolePrintf("dev eth get {addr} {bytecnt}\r");

            if(g_pEthernetSwitchHardwareResetFunc)
                DmConsolePrintf("dev eth reset\r");

            break;
        default:
            break;
    }

    return 0;
}

/**
 * \author      John Cheng
 * \date        7/15/2011
 * \return      none
 * \retval      none
 * \brief       To close all ethernet console socket.
 * \param       none
 */
void closeAllEthConsole(UINT8 reboot)
{
    // Find any console instance and close it
    for(int i=0; i<ETH_CONSOLE_MAX_CONNECTIONS; i++)
    {
        if(EthConsole.used[i])
        {
#if defined(DM_V24)
            DmConsolePrintf("Closing socket %d.\n",EthConsole.socket[i]);
            if(reboot)
                DmConsolePrintf("Rebooting...\r");
#endif
            NetClose(EthConsole.socket[i],true);
        }
    }
}

